<?php
/**
 * Analytics Dashboard Template
 * Displays visitor stats with Tailwind CSS styling and tabbed navigation
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

$current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
?>

<div class="wrap">
    <div class="max-w-7xl mx-auto py-8">

        <!-- Header -->
        <div class="mb-8">
            <h1 class="text-4xl font-bold text-gray-900 mb-2">Kantan Analytics</h1>
            <p class="text-gray-600">Showing data from the last <?php echo $days; ?> days</p>

            <!-- Time Period Selector -->
            <div class="mt-4 flex gap-2">
                <?php foreach ([7, 30, 90, 365] as $period): ?>
                    <a href="?page=kantan-analytics&days=<?php echo $period; ?>&tab=<?php echo esc_attr($current_tab); ?>"
                       class="px-4 py-2 rounded-lg <?php echo $days === $period ? 'bg-blue-600 text-white' : 'bg-white text-gray-700 hover:bg-gray-100 border'; ?> transition">
                        <?php echo $period === 365 ? '1 Year' : $period . ' Days'; ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Tab Navigation -->
        <div class="border-b border-gray-200 mb-8">
            <nav class="flex gap-8" aria-label="Tabs">
                <?php
                $tabs = [
                    'overview' => ['label' => 'Overview', 'icon' => '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>'],
                    'interactions' => ['label' => 'Interactions', 'icon' => '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 15l-2 5L9 9l11 4-5 2zm0 0l5 5M7.188 2.239l.777 2.897M5.136 7.965l-2.898-.777M13.95 4.05l-2.122 2.122m-5.657 5.656l-2.12 2.122"></path>'],
                    'journeys' => ['label' => 'Journeys', 'icon' => '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 20l-5.447-2.724A1 1 0 013 16.382V5.618a1 1 0 011.447-.894L9 7m0 13l6-3m-6 3V7m6 10l4.553 2.276A1 1 0 0021 18.382V7.618a1 1 0 00-.553-.894L15 4m0 13V4m0 0L9 7"></path>']
                ];
                foreach ($tabs as $tab_id => $tab_info):
                    $is_active = $current_tab === $tab_id;
                ?>
                    <a href="?page=kantan-analytics&days=<?php echo $days; ?>&tab=<?php echo $tab_id; ?>"
                       class="flex items-center gap-2 py-4 px-1 border-b-2 font-medium text-sm <?php echo $is_active ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> transition">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><?php echo $tab_info['icon']; ?></svg>
                        <?php echo $tab_info['label']; ?>
                    </a>
                <?php endforeach; ?>
            </nav>
        </div>

        <?php if ($current_tab === 'overview'): ?>
        <!-- ==================== OVERVIEW TAB ==================== -->

        <!-- Key Metrics Grid -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <!-- Total Visits -->
            <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-blue-500">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-600 text-sm font-medium uppercase">Total Visits</p>
                        <p class="text-3xl font-bold text-gray-900 mt-2"><?php echo number_format($total_visits); ?></p>
                    </div>
                    <div class="bg-blue-100 rounded-full p-3">
                        <svg class="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <!-- Avg Duration -->
            <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-green-500">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-600 text-sm font-medium uppercase">Avg Duration</p>
                        <p class="text-3xl font-bold text-gray-900 mt-2"><?php echo kantan_format_duration($avg_duration); ?></p>
                    </div>
                    <div class="bg-green-100 rounded-full p-3">
                        <svg class="w-8 h-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <!-- Bounce Rate -->
            <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-yellow-500">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-600 text-sm font-medium uppercase">Bounce Rate</p>
                        <p class="text-3xl font-bold text-gray-900 mt-2"><?php echo $bounce_rate; ?>%</p>
                    </div>
                    <div class="bg-yellow-100 rounded-full p-3">
                        <svg class="w-8 h-8 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <!-- Organic Traffic -->
            <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-purple-500">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-600 text-sm font-medium uppercase">Organic Traffic</p>
                        <p class="text-3xl font-bold text-gray-900 mt-2"><?php echo number_format($organic_count); ?></p>
                        <p class="text-sm text-gray-500 mt-1">vs <?php echo number_format($search_count); ?> other</p>
                    </div>
                    <div class="bg-purple-100 rounded-full p-3">
                        <svg class="w-8 h-8 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Visits Chart -->
        <div class="bg-white rounded-xl shadow-md p-6 mb-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Visits Over Time</h2>
            <canvas id="visitsChart" class="w-full" style="max-height: 300px;"></canvas>
        </div>

        <!-- Two Column Layout -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            <!-- Traffic Sources -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Traffic Sources</h2>
                <div class="space-y-4">
                    <?php if (empty($traffic_sources)): ?>
                        <p class="text-gray-500 italic">No traffic data yet</p>
                    <?php else: ?>
                        <?php foreach ($traffic_sources as $source):
                            $percentage = $total_visits > 0 ? round(($source->visits / $total_visits) * 100, 1) : 0;
                            $color_map = ['organic' => 'bg-green-500', 'direct' => 'bg-blue-500', 'referral' => 'bg-purple-500', 'social' => 'bg-pink-500'];
                            $color = $color_map[$source->traffic_source] ?? 'bg-gray-500';
                        ?>
                            <div>
                                <div class="flex justify-between items-center mb-1">
                                    <span class="text-gray-700 font-medium capitalize"><?php echo esc_html($source->traffic_source); ?></span>
                                    <span class="text-gray-600"><?php echo number_format($source->visits); ?> <span class="text-sm">(<?php echo $percentage; ?>%)</span></span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="<?php echo $color; ?> h-2 rounded-full" style="width: <?php echo $percentage; ?>%"></div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Top Referrers -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Top Referrers</h2>
                <div class="space-y-3">
                    <?php if (empty($top_referrers)): ?>
                        <p class="text-gray-500 italic">No referrer data yet</p>
                    <?php else: ?>
                        <?php foreach ($top_referrers as $referrer): ?>
                            <div class="referrer-row flex justify-between items-center py-3 border-b border-gray-100 last:border-0 cursor-pointer hover:bg-gray-50 rounded-lg px-2 -mx-2 transition"
                                 data-domain="<?php echo esc_attr($referrer->domain); ?>"
                                 title="Click to view details">
                                <div class="flex-1 min-w-0">
                                    <p class="text-sm text-gray-900 truncate"><?php echo esc_html($referrer->domain); ?></p>
                                    <p class="text-xs text-gray-500">Click for details</p>
                                </div>
                                <span class="ml-4 px-3 py-1 bg-blue-100 text-blue-800 text-sm font-medium rounded-full hover:bg-blue-200 transition"><?php echo number_format($referrer->visits); ?></span>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Referrer Details Modal -->
        <div id="referrer-modal" class="fixed inset-0 z-50 hidden">
            <div class="fixed inset-0 bg-black/50" id="referrer-modal-backdrop"></div>
            <div class="fixed inset-0 overflow-y-auto">
                <div class="flex min-h-full items-center justify-center p-4">
                    <div class="bg-white rounded-xl shadow-xl max-w-3xl w-full max-h-[80vh] overflow-hidden">
                        <!-- Modal Header -->
                        <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center bg-gray-50">
                            <h3 class="text-xl font-bold text-gray-900" id="referrer-modal-title">Referrer Details</h3>
                            <button id="referrer-modal-close" class="text-gray-400 hover:text-gray-600 transition">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </button>
                        </div>
                        <!-- Modal Body -->
                        <div class="p-6 overflow-y-auto max-h-[calc(80vh-80px)]" id="referrer-modal-body">
                            <div class="flex items-center justify-center py-8">
                                <svg class="animate-spin h-8 w-8 text-blue-600" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                    <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                    <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                </svg>
                                <span class="ml-3 text-gray-600">Loading...</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Top Pages -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Most Visited Pages</h2>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Page</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Views</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($top_pages)): ?>
                            <tr><td colspan="2" class="px-6 py-4 text-center text-gray-500 italic">No page data yet</td></tr>
                        <?php else: ?>
                            <?php foreach ($top_pages as $page): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4">
                                        <div class="text-sm font-medium text-gray-900"><?php echo esc_html($page->page_title ?: 'Untitled'); ?></div>
                                        <div class="text-sm text-gray-500 truncate" style="max-width: 500px;"><?php echo esc_html($page->page_url); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-3 py-1 inline-flex text-sm leading-5 font-semibold rounded-full bg-blue-100 text-blue-800"><?php echo number_format($page->views); ?></span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <?php elseif ($current_tab === 'interactions'): ?>
        <!-- ==================== INTERACTIONS TAB ==================== -->

        <!-- Click Tracking & Scroll Depth Section -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            <!-- Click Regions Heatmap -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <div class="flex justify-between items-center mb-6">
                    <h2 class="text-2xl font-bold text-gray-900">Click Regions</h2>
                    <span class="text-sm text-gray-500"><?php echo number_format($total_clicks); ?> total clicks</span>
                </div>
                <?php if ($total_clicks > 0):
                    $max_clicks = max($click_regions);
                    $region_labels = ['Top Left', 'Top Center', 'Top Right', 'Middle Left', 'Center', 'Middle Right', 'Bottom Left', 'Bottom Center', 'Bottom Right'];
                ?>
                    <div class="grid grid-cols-3 gap-2 mb-4">
                        <?php for ($i = 1; $i <= 9; $i++):
                            $clicks = $click_regions[$i];
                            $intensity = $max_clicks > 0 ? ($clicks / $max_clicks) : 0;
                            $bg_class = $intensity > 0.66 ? 'bg-red-500' : ($intensity > 0.33 ? 'bg-orange-400' : 'bg-blue-300');
                            if ($clicks == 0) $bg_class = 'bg-gray-100';
                        ?>
                            <div class="<?php echo $bg_class; ?> rounded-lg p-4 text-center transition-all hover:scale-105" title="<?php echo $region_labels[$i-1]; ?>">
                                <div class="text-lg font-bold <?php echo $clicks > 0 ? 'text-white' : 'text-gray-400'; ?>"><?php echo number_format($clicks); ?></div>
                                <div class="text-xs <?php echo $clicks > 0 ? 'text-white/80' : 'text-gray-400'; ?>"><?php echo $region_labels[$i-1]; ?></div>
                            </div>
                        <?php endfor; ?>
                    </div>
                    <p class="text-sm text-gray-500 text-center">Click density by page region</p>
                <?php else: ?>
                    <div class="text-center py-8">
                        <svg class="w-12 h-12 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 15l-2 5L9 9l11 4-5 2zm0 0l5 5M7.188 2.239l.777 2.897M5.136 7.965l-2.898-.777M13.95 4.05l-2.122 2.122m-5.657 5.656l-2.12 2.122"></path>
                        </svg>
                        <p class="text-gray-500 italic">No click data yet</p>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Scroll Depth -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <div class="flex justify-between items-center mb-6">
                    <h2 class="text-2xl font-bold text-gray-900">Scroll Depth</h2>
                    <span class="text-sm text-gray-500">Avg: <?php echo $avg_scroll_depth; ?>%</span>
                </div>
                <?php if ($scroll_distribution && $scroll_distribution->total > 0):
                    $depths = ['0-25%' => $scroll_distribution->depth_25, '26-50%' => $scroll_distribution->depth_50, '51-75%' => $scroll_distribution->depth_75, '76-100%' => $scroll_distribution->depth_100];
                    $colors = ['bg-red-400', 'bg-orange-400', 'bg-yellow-400', 'bg-green-500'];
                    $i = 0;
                ?>
                    <div class="space-y-4">
                        <?php foreach ($depths as $label => $count):
                            $percentage = round(($count / $scroll_distribution->total) * 100, 1);
                        ?>
                            <div>
                                <div class="flex justify-between items-center mb-1">
                                    <span class="text-gray-700 font-medium"><?php echo $label; ?></span>
                                    <span class="text-gray-600"><?php echo number_format($count); ?> <span class="text-sm">(<?php echo $percentage; ?>%)</span></span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-3">
                                    <div class="<?php echo $colors[$i]; ?> h-3 rounded-full transition-all" style="width: <?php echo $percentage; ?>%"></div>
                                </div>
                            </div>
                        <?php $i++; endforeach; ?>
                    </div>
                    <p class="text-sm text-gray-500 text-center mt-4">How far users scroll down pages</p>
                <?php else: ?>
                    <div class="text-center py-8">
                        <svg class="w-12 h-12 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 14l-7 7m0 0l-7-7m7 7V3"></path>
                        </svg>
                        <p class="text-gray-500 italic">No scroll data yet</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Top Clicked Elements -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Top Clicked Elements</h2>
            <?php if (empty($top_clicked_elements)): ?>
                <p class="text-gray-500 italic text-center py-8">No click data yet</p>
            <?php else: ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Element</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Clicks</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($top_clicked_elements as $element): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4">
                                        <div class="text-sm text-gray-900 truncate" style="max-width: 400px;"><?php echo esc_html($element->element_text ?: '(no text)'); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 py-1 text-xs font-mono bg-gray-100 text-gray-700 rounded">&lt;<?php echo esc_html($element->element_tag); ?>&gt;</span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-3 py-1 inline-flex text-sm leading-5 font-semibold rounded-full bg-orange-100 text-orange-800"><?php echo number_format($element->clicks); ?></span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>

        <?php elseif ($current_tab === 'journeys'): ?>
        <!-- ==================== JOURNEYS TAB ==================== -->

        <!-- User Journeys Section -->
        <div class="bg-white rounded-xl shadow-md p-6 mb-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Recent User Journeys</h2>
            <?php if (empty($session_journeys)): ?>
                <div class="text-center py-8">
                    <svg class="w-12 h-12 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 20l-5.447-2.724A1 1 0 013 16.382V5.618a1 1 0 011.447-.894L9 7m0 13l6-3m-6 3V7m6 10l4.553 2.276A1 1 0 0021 18.382V7.618a1 1 0 00-.553-.894L15 4m0 13V4m0 0L9 7"></path>
                    </svg>
                    <p class="text-gray-500 italic">No journey data yet</p>
                </div>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($session_journeys as $journey):
                        $source_colors = ['organic' => 'bg-green-100 text-green-800', 'direct' => 'bg-blue-100 text-blue-800', 'referral' => 'bg-purple-100 text-purple-800', 'social' => 'bg-pink-100 text-pink-800'];
                        $source_color = $source_colors[$journey->traffic_source] ?? 'bg-gray-100 text-gray-800';
                    ?>
                        <div class="border-l-4 border-blue-500 pl-4 py-3 hover:bg-gray-50 rounded-r-lg transition">
                            <div class="flex items-center gap-3 text-sm text-gray-500 mb-2">
                                <span><?php echo date('M j, g:i A', strtotime($journey->session_start)); ?></span>
                                <span class="<?php echo $source_color; ?> px-2 py-0.5 rounded text-xs font-medium capitalize"><?php echo esc_html($journey->traffic_source ?: 'unknown'); ?></span>
                                <span><?php echo $journey->page_count; ?> page<?php echo $journey->page_count != 1 ? 's' : ''; ?></span>
                                <span><?php echo kantan_format_duration($journey->total_duration ?: 0); ?></span>
                            </div>
                            <div class="flex items-center gap-1 flex-wrap">
                                <?php foreach ($journey->pages as $index => $page):
                                    $is_first = $index === 0;
                                    $is_last = $index === count($journey->pages) - 1;
                                    $page_class = $is_first ? 'bg-green-100 text-green-800' : ($is_last ? 'bg-red-100 text-red-800' : 'bg-gray-100 text-gray-700');
                                    $path = parse_url($page->page_url, PHP_URL_PATH) ?: '/';
                                    if (strlen($path) > 30) $path = '...' . substr($path, -27);
                                ?>
                                    <?php if ($index > 0): ?><span class="text-gray-400">→</span><?php endif; ?>
                                    <span class="<?php echo $page_class; ?> px-2 py-1 rounded text-sm" title="<?php echo esc_attr($page->page_title ?: $page->page_url); ?>"><?php echo esc_html($path); ?></span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- Page Flow Section -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- Page Transitions -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Top Page Transitions</h2>
                <?php if (empty($page_transitions)): ?>
                    <p class="text-gray-500 italic text-center py-8">No transition data yet</p>
                <?php else: ?>
                    <div class="space-y-3">
                        <?php foreach ($page_transitions as $transition):
                            $from_path = parse_url($transition->from_page, PHP_URL_PATH) ?: '/';
                            $to_path = parse_url($transition->to_page, PHP_URL_PATH) ?: '/';
                            if (strlen($from_path) > 20) $from_path = '...' . substr($from_path, -17);
                            if (strlen($to_path) > 20) $to_path = '...' . substr($to_path, -17);
                        ?>
                            <div class="flex items-center justify-between py-2 border-b border-gray-100 last:border-0">
                                <div class="flex items-center gap-2 flex-1 min-w-0">
                                    <span class="bg-blue-100 text-blue-800 px-2 py-1 rounded text-sm truncate" title="<?php echo esc_attr($transition->from_page); ?>"><?php echo esc_html($from_path); ?></span>
                                    <span class="text-gray-400">→</span>
                                    <span class="bg-gray-100 text-gray-700 px-2 py-1 rounded text-sm truncate" title="<?php echo esc_attr($transition->to_page); ?>"><?php echo esc_html($to_path); ?></span>
                                </div>
                                <span class="ml-2 px-2 py-1 bg-purple-100 text-purple-800 text-sm font-medium rounded"><?php echo number_format($transition->transitions); ?></span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Entry & Exit Pages -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Entry & Exit Pages</h2>
                <div class="grid grid-cols-2 gap-6">
                    <!-- Entry Pages -->
                    <div>
                        <h3 class="text-sm font-semibold text-gray-600 uppercase mb-3 flex items-center gap-2">
                            <span class="w-2 h-2 bg-green-500 rounded-full"></span>
                            Entry Pages
                        </h3>
                        <?php if (empty($entry_pages)): ?>
                            <p class="text-gray-500 italic text-sm">No data</p>
                        <?php else: ?>
                            <div class="space-y-2">
                                <?php foreach ($entry_pages as $page):
                                    $path = parse_url($page->page_url, PHP_URL_PATH) ?: '/';
                                    if (strlen($path) > 20) $path = '...' . substr($path, -17);
                                ?>
                                    <div class="flex justify-between items-center text-sm">
                                        <span class="text-gray-700 truncate" title="<?php echo esc_attr($page->page_url); ?>"><?php echo esc_html($path); ?></span>
                                        <span class="text-gray-500 ml-2"><?php echo number_format($page->entries); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    <!-- Exit Pages -->
                    <div>
                        <h3 class="text-sm font-semibold text-gray-600 uppercase mb-3 flex items-center gap-2">
                            <span class="w-2 h-2 bg-red-500 rounded-full"></span>
                            Exit Pages
                        </h3>
                        <?php if (empty($exit_pages)): ?>
                            <p class="text-gray-500 italic text-sm">No data</p>
                        <?php else: ?>
                            <div class="space-y-2">
                                <?php foreach ($exit_pages as $page):
                                    $path = parse_url($page->page_url, PHP_URL_PATH) ?: '/';
                                    if (strlen($path) > 20) $path = '...' . substr($path, -17);
                                ?>
                                    <div class="flex justify-between items-center text-sm">
                                        <span class="text-gray-700 truncate" title="<?php echo esc_attr($page->page_url); ?>"><?php echo esc_html($path); ?></span>
                                        <span class="text-gray-500 ml-2"><?php echo number_format($page->exits); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <?php endif; ?>

    </div>
</div>

<!-- Chart.js Script (only needed for overview tab) -->
<?php if ($current_tab === 'overview'): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const ctx = document.getElementById('visitsChart');
    if (ctx) {
        const visitsData = <?php echo json_encode(array_map(function($v) { return $v->visits; }, $visits_by_day)); ?>;
        const visitsDates = <?php echo json_encode(array_map(function($v) { return date('M j', strtotime($v->date)); }, $visits_by_day)); ?>;

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: visitsDates,
                datasets: [{
                    label: 'Visits',
                    data: visitsData,
                    borderColor: 'rgb(59, 130, 246)',
                    backgroundColor: 'rgba(59, 130, 246, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } },
                scales: { y: { beginAtZero: true, ticks: { precision: 0 } } }
            }
        });
    }

    // Referrer drill-down modal functionality
    const modal = document.getElementById('referrer-modal');
    const modalTitle = document.getElementById('referrer-modal-title');
    const modalBody = document.getElementById('referrer-modal-body');
    const modalClose = document.getElementById('referrer-modal-close');
    const modalBackdrop = document.getElementById('referrer-modal-backdrop');
    const days = <?php echo intval($days); ?>;

    function openModal(domain) {
        modal.classList.remove('hidden');
        document.body.style.overflow = 'hidden';
        modalTitle.textContent = 'Traffic from ' + domain;
        modalBody.innerHTML = `
            <div class="flex items-center justify-center py-8">
                <svg class="animate-spin h-8 w-8 text-blue-600" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                    <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>
                <span class="ml-3 text-gray-600">Loading...</span>
            </div>
        `;

        // Fetch referrer details
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'kantan_get_referrer_details',
                domain: domain,
                days: days
            })
        })
        .then(response => response.json())
        .then(result => {
            if (result.success) {
                renderModalContent(result.data);
            } else {
                modalBody.innerHTML = '<p class="text-red-500 text-center py-8">Error loading data</p>';
            }
        })
        .catch(error => {
            modalBody.innerHTML = '<p class="text-red-500 text-center py-8">Error loading data</p>';
        });
    }

    function renderModalContent(data) {
        let html = `
            <div class="mb-6 p-4 bg-blue-50 rounded-lg">
                <p class="text-lg font-semibold text-blue-900">${data.total_visits} total visits from ${escapeHtml(data.domain)}</p>
            </div>
        `;

        // Pages visited section
        html += `
            <div class="mb-8">
                <h4 class="text-lg font-bold text-gray-900 mb-4 flex items-center gap-2">
                    <svg class="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                    Pages Visited
                </h4>
        `;

        if (data.pages_visited && data.pages_visited.length > 0) {
            html += '<div class="space-y-2">';
            data.pages_visited.forEach(page => {
                const path = new URL(page.page_url, window.location.origin).pathname || '/';
                html += `
                    <div class="flex justify-between items-center py-2 px-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition">
                        <div class="flex-1 min-w-0 mr-4">
                            <p class="text-sm font-medium text-gray-900 truncate">${escapeHtml(page.page_title || 'Untitled')}</p>
                            <p class="text-xs text-gray-500 truncate">${escapeHtml(path)}</p>
                        </div>
                        <span class="px-2 py-1 bg-green-100 text-green-800 text-sm font-medium rounded">${page.visits}</span>
                    </div>
                `;
            });
            html += '</div>';
        } else {
            html += '<p class="text-gray-500 italic">No page data available</p>';
        }
        html += '</div>';

        // Referrer URLs section
        html += `
            <div>
                <h4 class="text-lg font-bold text-gray-900 mb-4 flex items-center gap-2">
                    <svg class="w-5 h-5 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1"></path>
                    </svg>
                    Referring URLs
                </h4>
        `;

        if (data.referrer_urls && data.referrer_urls.length > 0) {
            html += '<div class="space-y-2">';
            data.referrer_urls.forEach(ref => {
                let displayUrl = ref.referrer;
                try {
                    const url = new URL(ref.referrer);
                    displayUrl = url.pathname + url.search;
                    if (displayUrl === '/') displayUrl = '(homepage)';
                } catch(e) {}

                html += `
                    <div class="flex justify-between items-center py-2 px-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition">
                        <div class="flex-1 min-w-0 mr-4">
                            <a href="${escapeHtml(ref.referrer)}" target="_blank" rel="noopener noreferrer"
                               class="text-sm text-blue-600 hover:text-blue-800 hover:underline truncate block"
                               title="${escapeHtml(ref.referrer)}">
                                ${escapeHtml(displayUrl)}
                            </a>
                        </div>
                        <span class="px-2 py-1 bg-purple-100 text-purple-800 text-sm font-medium rounded">${ref.visits}</span>
                    </div>
                `;
            });
            html += '</div>';
        } else {
            html += '<p class="text-gray-500 italic">No referrer URL data available</p>';
        }
        html += '</div>';

        modalBody.innerHTML = html;
    }

    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    function closeModal() {
        modal.classList.add('hidden');
        document.body.style.overflow = '';
    }

    // Event listeners for referrer rows
    document.querySelectorAll('.referrer-row').forEach(row => {
        row.addEventListener('click', function() {
            const domain = this.getAttribute('data-domain');
            if (domain) openModal(domain);
        });
    });

    // Close modal events
    modalClose.addEventListener('click', closeModal);
    modalBackdrop.addEventListener('click', closeModal);
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && !modal.classList.contains('hidden')) {
            closeModal();
        }
    });
});
</script>
<?php endif; ?>
