<?php
/**
 * Plugin Name: Kantan Analytics
 * Plugin URI: https://plugins.joeljenkins.me
 * Description: Lightweight, privacy-friendly analytics for WordPress that keeps all data in your database. Track visits, clicks, scroll depth, user journeys, and traffic sources with a beautiful Tailwind dashboard. No external services, no cookies - just simple, self-hosted visitor insights.
 * Version: 2.0.2
 * Author: Kinda Useful Plugins
 * Author URI: https://plugins.joeljenkins.me
 * Requires at least: 5.0
 * Requires PHP: 7.0
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: kantan-analytics
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('KANTAN_ANALYTICS_VERSION', '2.0.2');
define('KANTAN_ANALYTICS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('KANTAN_ANALYTICS_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Load plugin modules
 */
require_once KANTAN_ANALYTICS_PLUGIN_DIR . 'includes/database.php';
require_once KANTAN_ANALYTICS_PLUGIN_DIR . 'includes/api.php';
require_once KANTAN_ANALYTICS_PLUGIN_DIR . 'includes/analytics.php';
require_once KANTAN_ANALYTICS_PLUGIN_DIR . 'includes/admin.php';

// Auto-updates from Kinda Useful Plugins
require_once KANTAN_ANALYTICS_PLUGIN_DIR . 'includes/class-plugin-updater.php';
new Kinda_Useful_Plugin_Updater(
    'https://plugins.joeljenkins.me',
    __FILE__,
    'kantan-analytics'
);

/**
 * Activation hook - create database tables and schedule cleanup
 */
function kantan_analytics_activate() {
    kantan_analytics_create_tables();

    // Schedule daily cleanup if not already scheduled
    if (!wp_next_scheduled('kantan_analytics_daily_cleanup')) {
        wp_schedule_event(time(), 'daily', 'kantan_analytics_daily_cleanup');
    }
}
register_activation_hook(__FILE__, 'kantan_analytics_activate');

/**
 * Deactivation hook - clear scheduled events
 */
function kantan_analytics_deactivate() {
    wp_clear_scheduled_hook('kantan_analytics_daily_cleanup');
}
register_deactivation_hook(__FILE__, 'kantan_analytics_deactivate');

/**
 * Daily cleanup - remove old analytics data based on retention settings
 */
function kantan_analytics_cleanup_old_data() {
    global $wpdb;

    // Default retention: 90 days (can be made configurable via settings page later)
    $retention_days = apply_filters('kantan_analytics_retention_days', 90);
    $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$retention_days} days"));

    // Clean up main analytics table
    $analytics_table = $wpdb->prefix . 'kantan_analytics';
    $wpdb->query($wpdb->prepare(
        "DELETE FROM $analytics_table WHERE visit_date < %s",
        $cutoff_date
    ));

    // Clean up clicks table
    $clicks_table = $wpdb->prefix . 'kantan_clicks';
    $wpdb->query($wpdb->prepare(
        "DELETE FROM $clicks_table WHERE click_time < %s",
        $cutoff_date
    ));

    // Clean up pageviews table
    $pageviews_table = $wpdb->prefix . 'kantan_pageviews';
    $wpdb->query($wpdb->prepare(
        "DELETE FROM $pageviews_table WHERE entered_at < %s",
        $cutoff_date
    ));
}
add_action('kantan_analytics_daily_cleanup', 'kantan_analytics_cleanup_old_data');
