<?php

/**
 * Analytics Database Setup
 * Creates and manages database tables for tracking visitor analytics
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/* Create analytics database table */
function kantan_analytics_create_tables() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        session_id varchar(255) NOT NULL,
        page_url varchar(500) NOT NULL,
        page_title varchar(255) DEFAULT NULL,
        referrer varchar(500) DEFAULT NULL,
        traffic_source varchar(50) DEFAULT NULL,
        is_bounce tinyint(1) DEFAULT 1,
        visit_duration int(11) DEFAULT 0,
        screen_width int(11) DEFAULT NULL,
        screen_height int(11) DEFAULT NULL,
        user_agent varchar(500) DEFAULT NULL,
        ip_address varchar(45) DEFAULT NULL,
        visit_date datetime DEFAULT CURRENT_TIMESTAMP,
        last_seen datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY  (id),
        KEY session_id (session_id),
        KEY visit_date (visit_date),
        KEY last_seen (last_seen),
        KEY traffic_source (traffic_source)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    // Add last_seen column if it doesn't exist (for existing installations)
    $column_exists = $wpdb->get_results("SHOW COLUMNS FROM $table_name LIKE 'last_seen'");
    if (empty($column_exists)) {
        $wpdb->query("ALTER TABLE $table_name ADD COLUMN last_seen datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP AFTER visit_date");
        $wpdb->query("ALTER TABLE $table_name ADD KEY last_seen (last_seen)");
    }

    // Create clicks and pageviews tables (v2.0.0)
    kantan_analytics_create_clicks_table();
    kantan_analytics_create_pageviews_table();
}

/**
 * Create clicks tracking table
 * Stores individual click events with position data
 */
function kantan_analytics_create_clicks_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        session_id varchar(255) NOT NULL,
        page_url varchar(500) NOT NULL,
        click_x_pct tinyint(3) UNSIGNED NOT NULL,
        click_y_pct tinyint(3) UNSIGNED NOT NULL,
        click_region tinyint(1) UNSIGNED NOT NULL,
        element_tag varchar(50) DEFAULT NULL,
        element_text varchar(255) DEFAULT NULL,
        click_time datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id),
        KEY session_id (session_id),
        KEY page_url (page_url(191)),
        KEY click_time (click_time),
        KEY click_region (click_region)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

/**
 * Create pageviews tracking table
 * Stores individual page views with sequence and timing data for journey tracking
 */
function kantan_analytics_create_pageviews_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        session_id varchar(255) NOT NULL,
        page_url varchar(500) NOT NULL,
        page_title varchar(255) DEFAULT NULL,
        sequence_num smallint(5) UNSIGNED NOT NULL,
        scroll_depth tinyint(3) UNSIGNED DEFAULT 0,
        time_on_page int(11) DEFAULT 0,
        entered_at datetime DEFAULT CURRENT_TIMESTAMP,
        exited_at datetime DEFAULT NULL,
        PRIMARY KEY  (id),
        KEY session_id (session_id),
        KEY entered_at (entered_at),
        UNIQUE KEY session_sequence (session_id, sequence_num)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
