<?php

/**
 * Analytics API Endpoint
 * Handles AJAX requests for tracking visitor data
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/* Handle analytics tracking via AJAX */
function kantan_track_visit() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';
    $pageviews_table = $wpdb->prefix . 'kantan_pageviews';

    // Get POST data
    $session_id = sanitize_text_field($_POST['session_id'] ?? '');
    $page_url = esc_url_raw($_POST['page_url'] ?? '');
    $page_title = sanitize_text_field($_POST['page_title'] ?? '');
    $referrer = esc_url_raw($_POST['referrer'] ?? '');
    $traffic_source = sanitize_text_field($_POST['traffic_source'] ?? 'direct');
    $is_bounce = intval($_POST['is_bounce'] ?? 1);
    $visit_duration = intval($_POST['visit_duration'] ?? 0);
    $screen_width = intval($_POST['screen_width'] ?? 0);
    $screen_height = intval($_POST['screen_height'] ?? 0);
    $sequence_num = intval($_POST['sequence_num'] ?? 1);
    $scroll_depth = intval($_POST['scroll_depth'] ?? 0);

    // Get server data
    $user_agent = sanitize_text_field($_SERVER['HTTP_USER_AGENT'] ?? '');
    $ip_address = sanitize_text_field($_SERVER['REMOTE_ADDR'] ?? '');

    // Check if this session already has a record in main analytics table
    $existing = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM $table_name WHERE session_id = %s ORDER BY id DESC LIMIT 1",
        $session_id
    ));

    if ($existing) {
        // Update existing record
        $wpdb->update(
            $table_name,
            array(
                'is_bounce' => $is_bounce,
                'visit_duration' => $visit_duration,
                'page_url' => $page_url,
                'page_title' => $page_title
            ),
            array('id' => $existing->id)
        );
    } else {
        // Insert new record
        $wpdb->insert(
            $table_name,
            array(
                'session_id' => $session_id,
                'page_url' => $page_url,
                'page_title' => $page_title,
                'referrer' => $referrer,
                'traffic_source' => $traffic_source,
                'is_bounce' => $is_bounce,
                'visit_duration' => $visit_duration,
                'screen_width' => $screen_width,
                'screen_height' => $screen_height,
                'user_agent' => $user_agent,
                'ip_address' => $ip_address
            )
        );
    }

    // Track pageview in pageviews table for journey tracking
    $existing_pageview = $wpdb->get_row($wpdb->prepare(
        "SELECT id, time_on_page FROM $pageviews_table WHERE session_id = %s AND sequence_num = %d",
        $session_id,
        $sequence_num
    ));

    if ($existing_pageview) {
        // Update existing pageview with duration and scroll depth
        $wpdb->update(
            $pageviews_table,
            array(
                'time_on_page' => $visit_duration,
                'scroll_depth' => $scroll_depth,
                'exited_at' => current_time('mysql')
            ),
            array('id' => $existing_pageview->id)
        );
    } else {
        // Insert new pageview record
        $wpdb->insert(
            $pageviews_table,
            array(
                'session_id' => $session_id,
                'page_url' => $page_url,
                'page_title' => $page_title,
                'sequence_num' => $sequence_num,
                'scroll_depth' => $scroll_depth,
                'time_on_page' => $visit_duration
            )
        );

        // Update previous page's exit time if this is not the first page
        if ($sequence_num > 1) {
            $wpdb->query($wpdb->prepare(
                "UPDATE $pageviews_table SET exited_at = %s WHERE session_id = %s AND sequence_num = %d AND exited_at IS NULL",
                current_time('mysql'),
                $session_id,
                $sequence_num - 1
            ));
        }
    }

    wp_send_json_success();
}

/* Handle click tracking via AJAX */
function kantan_track_clicks() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';

    // Get POST data
    $session_id = sanitize_text_field($_POST['session_id'] ?? '');
    $page_url = esc_url_raw($_POST['page_url'] ?? '');
    $clicks_json = $_POST['clicks'] ?? '[]';

    // Decode clicks array
    $clicks = json_decode(stripslashes($clicks_json), true);

    if (!is_array($clicks) || empty($clicks)) {
        wp_send_json_success();
        return;
    }

    // Insert each click
    foreach ($clicks as $click) {
        $wpdb->insert(
            $table_name,
            array(
                'session_id' => $session_id,
                'page_url' => $page_url,
                'click_x_pct' => intval($click['x'] ?? 0),
                'click_y_pct' => intval($click['y'] ?? 0),
                'click_region' => intval($click['region'] ?? 5),
                'element_tag' => sanitize_text_field($click['tag'] ?? ''),
                'element_text' => sanitize_text_field(substr($click['text'] ?? '', 0, 255))
            )
        );
    }

    wp_send_json_success();
}

/* Get referrer details for drill-down modal (admin only) */
function kantan_get_referrer_details() {
    // Check for admin permissions
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
        return;
    }

    $domain = sanitize_text_field($_POST['domain'] ?? '');
    $days = intval($_POST['days'] ?? 30);

    if (empty($domain)) {
        wp_send_json_error('Missing domain parameter');
        return;
    }

    $details = kantan_get_referrer_domain_details($domain, $days);
    wp_send_json_success($details);
}

/* Register AJAX handlers for both logged in and logged out users */
add_action('wp_ajax_kantan_track_visit', 'kantan_track_visit');
add_action('wp_ajax_nopriv_kantan_track_visit', 'kantan_track_visit');
add_action('wp_ajax_kantan_track_clicks', 'kantan_track_clicks');
add_action('wp_ajax_nopriv_kantan_track_clicks', 'kantan_track_clicks');
add_action('wp_ajax_kantan_get_referrer_details', 'kantan_get_referrer_details');

/* Enqueue tracking JavaScript */
function kantan_enqueue_tracking_script() {
    wp_enqueue_script(
        'kantan-analytics',
        KANTAN_ANALYTICS_PLUGIN_URL . 'assets/tracking.js',
        array(),
        KANTAN_ANALYTICS_VERSION,
        true
    );

    // Pass AJAX URL to JavaScript
    wp_localize_script('kantan-analytics', 'kantanAnalytics', array(
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('kantan_analytics_nonce')
    ));
}
add_action('wp_enqueue_scripts', 'kantan_enqueue_tracking_script');
