<?php

/**
 * Analytics Calculation Functions
 * Retrieves and processes analytics data from database
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/* Get total visits for a date range */
function kantan_get_total_visits($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(DISTINCT session_id) FROM $table_name WHERE visit_date >= %s",
        $date
    ));
}

/* Get average session duration in seconds */
function kantan_get_avg_duration($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    $avg = $wpdb->get_var($wpdb->prepare(
        "SELECT AVG(visit_duration) FROM $table_name WHERE visit_date >= %s AND visit_duration > 0",
        $date
    ));

    return round($avg ?: 0);
}

/* Get bounce rate percentage */
function kantan_get_bounce_rate($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    $total = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(DISTINCT session_id) FROM $table_name WHERE visit_date >= %s",
        $date
    ));

    $bounces = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(DISTINCT session_id) FROM $table_name WHERE visit_date >= %s AND is_bounce = 1",
        $date
    ));

    if ($total == 0) return 0;

    return round(($bounces / $total) * 100, 1);
}

/* Get traffic sources breakdown */
function kantan_get_traffic_sources($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT traffic_source, COUNT(DISTINCT session_id) as visits
        FROM $table_name
        WHERE visit_date >= %s
        GROUP BY traffic_source
        ORDER BY visits DESC",
        $date
    ));
}

/* Get top referrers */
function kantan_get_top_referrers($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT referrer, COUNT(DISTINCT session_id) as visits
        FROM $table_name
        WHERE visit_date >= %s
        AND referrer != ''
        AND referrer IS NOT NULL
        GROUP BY referrer
        ORDER BY visits DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Get visits by day for chart */
function kantan_get_visits_by_day($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT DATE(visit_date) as date, COUNT(DISTINCT session_id) as visits
        FROM $table_name
        WHERE visit_date >= %s
        GROUP BY DATE(visit_date)
        ORDER BY date ASC",
        $date
    ));
}

/* Get most visited pages */
function kantan_get_top_pages($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT page_url, page_title, COUNT(*) as views
        FROM $table_name
        WHERE visit_date >= %s
        GROUP BY page_url
        ORDER BY views DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Format duration in human readable format */
function kantan_format_duration($seconds) {
    if ($seconds < 60) {
        return $seconds . 's';
    } elseif ($seconds < 3600) {
        return floor($seconds / 60) . 'm ' . ($seconds % 60) . 's';
    } else {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        return $hours . 'h ' . $minutes . 'm';
    }
}

/* ========================================
   CLICK TRACKING FUNCTIONS
   ======================================== */

/* Get click heatmap data for a specific page */
function kantan_get_click_heatmap($page_url = '', $days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    if ($page_url) {
        return $wpdb->get_results($wpdb->prepare(
            "SELECT click_x_pct, click_y_pct, COUNT(*) as clicks
            FROM $table_name
            WHERE click_time >= %s AND page_url = %s
            GROUP BY click_x_pct, click_y_pct
            ORDER BY clicks DESC",
            $date,
            $page_url
        ));
    }

    return $wpdb->get_results($wpdb->prepare(
        "SELECT click_x_pct, click_y_pct, COUNT(*) as clicks
        FROM $table_name
        WHERE click_time >= %s
        GROUP BY click_x_pct, click_y_pct
        ORDER BY clicks DESC",
        $date
    ));
}

/* Get click region summary (3x3 grid) */
function kantan_get_click_regions($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    $results = $wpdb->get_results($wpdb->prepare(
        "SELECT click_region, COUNT(*) as clicks
        FROM $table_name
        WHERE click_time >= %s
        GROUP BY click_region
        ORDER BY click_region ASC",
        $date
    ));

    // Initialize all 9 regions with 0
    $regions = array_fill(1, 9, 0);
    foreach ($results as $row) {
        $regions[$row->click_region] = intval($row->clicks);
    }

    return $regions;
}

/* Get total clicks count */
function kantan_get_total_clicks($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE click_time >= %s",
        $date
    )) ?: 0;
}

/* Get top clicked elements */
function kantan_get_top_clicked_elements($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT element_tag, element_text, COUNT(*) as clicks
        FROM $table_name
        WHERE click_time >= %s AND element_text != ''
        GROUP BY element_tag, element_text
        ORDER BY clicks DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Get pages with click data */
function kantan_get_pages_with_clicks($days = 30, $limit = 20) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_clicks';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT page_url, COUNT(*) as clicks
        FROM $table_name
        WHERE click_time >= %s
        GROUP BY page_url
        ORDER BY clicks DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* ========================================
   JOURNEY TRACKING FUNCTIONS
   ======================================== */

/* Get session journeys (page sequences) */
function kantan_get_session_journeys($days = 30, $limit = 20) {
    global $wpdb;
    $pageviews_table = $wpdb->prefix . 'kantan_pageviews';
    $analytics_table = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    // Get unique sessions with their details
    $sessions = $wpdb->get_results($wpdb->prepare(
        "SELECT DISTINCT p.session_id,
            MIN(p.entered_at) as session_start,
            MAX(p.sequence_num) as page_count,
            SUM(p.time_on_page) as total_duration,
            a.traffic_source
        FROM $pageviews_table p
        LEFT JOIN $analytics_table a ON p.session_id = a.session_id
        WHERE p.entered_at >= %s
        GROUP BY p.session_id
        ORDER BY session_start DESC
        LIMIT %d",
        $date,
        $limit
    ));

    // For each session, get the page sequence
    foreach ($sessions as &$session) {
        $session->pages = $wpdb->get_results($wpdb->prepare(
            "SELECT page_url, page_title, sequence_num, time_on_page, scroll_depth
            FROM $pageviews_table
            WHERE session_id = %s
            ORDER BY sequence_num ASC",
            $session->session_id
        ));
    }

    return $sessions;
}

/* Get page transitions (from -> to) */
function kantan_get_page_transitions($days = 30, $limit = 20) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT
            p1.page_url as from_page,
            p2.page_url as to_page,
            COUNT(*) as transitions
        FROM $table_name p1
        INNER JOIN $table_name p2 ON p1.session_id = p2.session_id
            AND p2.sequence_num = p1.sequence_num + 1
        WHERE p1.entered_at >= %s
        GROUP BY p1.page_url, p2.page_url
        ORDER BY transitions DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Get entry pages (first page in session) */
function kantan_get_entry_pages($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT page_url, page_title, COUNT(*) as entries
        FROM $table_name
        WHERE entered_at >= %s AND sequence_num = 1
        GROUP BY page_url
        ORDER BY entries DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Get exit pages (last page in session) */
function kantan_get_exit_pages($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT p.page_url, p.page_title, COUNT(*) as exits
        FROM $table_name p
        INNER JOIN (
            SELECT session_id, MAX(sequence_num) as max_seq
            FROM $table_name
            WHERE entered_at >= %s
            GROUP BY session_id
        ) m ON p.session_id = m.session_id AND p.sequence_num = m.max_seq
        WHERE p.entered_at >= %s
        GROUP BY p.page_url
        ORDER BY exits DESC
        LIMIT %d",
        $date,
        $date,
        $limit
    ));
}

/* ========================================
   SCROLL DEPTH FUNCTIONS
   ======================================== */

/* Get average scroll depth by page */
function kantan_get_scroll_depth_by_page($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    return $wpdb->get_results($wpdb->prepare(
        "SELECT page_url, page_title, AVG(scroll_depth) as avg_scroll, COUNT(*) as views
        FROM $table_name
        WHERE entered_at >= %s AND scroll_depth > 0
        GROUP BY page_url
        ORDER BY views DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Get scroll depth distribution for a page */
function kantan_get_scroll_distribution($page_url = '', $days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    // Get counts for each quartile (0-25, 26-50, 51-75, 76-100)
    $where_page = $page_url ? $wpdb->prepare(" AND page_url = %s", $page_url) : "";

    $results = $wpdb->get_results($wpdb->prepare(
        "SELECT
            SUM(CASE WHEN scroll_depth <= 25 THEN 1 ELSE 0 END) as depth_25,
            SUM(CASE WHEN scroll_depth > 25 AND scroll_depth <= 50 THEN 1 ELSE 0 END) as depth_50,
            SUM(CASE WHEN scroll_depth > 50 AND scroll_depth <= 75 THEN 1 ELSE 0 END) as depth_75,
            SUM(CASE WHEN scroll_depth > 75 THEN 1 ELSE 0 END) as depth_100,
            COUNT(*) as total
        FROM $table_name
        WHERE entered_at >= %s AND scroll_depth > 0" . $where_page,
        $date
    ));

    return $results ? $results[0] : null;
}

/* Get average scroll depth across all pages */
function kantan_get_avg_scroll_depth($days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    $avg = $wpdb->get_var($wpdb->prepare(
        "SELECT AVG(scroll_depth) FROM $table_name WHERE entered_at >= %s AND scroll_depth > 0",
        $date
    ));

    return round($avg ?: 0);
}

/* ========================================
   REFERRER DRILL-DOWN FUNCTIONS
   ======================================== */

/* Get top referrer domains (grouped by domain) */
function kantan_get_top_referrer_domains($days = 30, $limit = 10) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    // Use SUBSTRING_INDEX to extract domain from referrer URL
    return $wpdb->get_results($wpdb->prepare(
        "SELECT
            SUBSTRING_INDEX(SUBSTRING_INDEX(REPLACE(REPLACE(referrer, 'https://', ''), 'http://', ''), '/', 1), '?', 1) as domain,
            COUNT(DISTINCT session_id) as visits
        FROM $table_name
        WHERE visit_date >= %s
        AND referrer != ''
        AND referrer IS NOT NULL
        GROUP BY domain
        ORDER BY visits DESC
        LIMIT %d",
        $date,
        $limit
    ));
}

/* Get referrer details for a specific domain */
function kantan_get_referrer_domain_details($domain, $days = 30) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'kantan_analytics';
    $pageviews_table = $wpdb->prefix . 'kantan_pageviews';

    $date = date('Y-m-d H:i:s', strtotime("-$days days"));

    // Escape domain for LIKE query
    $domain_like = '%' . $wpdb->esc_like($domain) . '%';

    // Get full referrer URLs from this domain
    $referrer_urls = $wpdb->get_results($wpdb->prepare(
        "SELECT referrer, COUNT(DISTINCT session_id) as visits
        FROM $table_name
        WHERE visit_date >= %s
        AND referrer LIKE %s
        GROUP BY referrer
        ORDER BY visits DESC
        LIMIT 20",
        $date,
        $domain_like
    ));

    // Get pages visited from this referrer domain
    $pages_visited = $wpdb->get_results($wpdb->prepare(
        "SELECT a.page_url, a.page_title, COUNT(DISTINCT a.session_id) as visits
        FROM $table_name a
        WHERE a.visit_date >= %s
        AND a.referrer LIKE %s
        GROUP BY a.page_url
        ORDER BY visits DESC
        LIMIT 20",
        $date,
        $domain_like
    ));

    // Get total visits from this domain
    $total_visits = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(DISTINCT session_id)
        FROM $table_name
        WHERE visit_date >= %s
        AND referrer LIKE %s",
        $date,
        $domain_like
    ));

    return array(
        'domain' => $domain,
        'total_visits' => intval($total_visits),
        'referrer_urls' => $referrer_urls,
        'pages_visited' => $pages_visited
    );
}
