/**
 * Kantan Analytics Tracking
 * Captures visitor data, clicks, scroll depth, and page journeys
 */

(function() {
    'use strict';

    // Generate or retrieve session ID
    function getSessionId() {
        let sessionId = sessionStorage.getItem('kantan_session_id');
        if (!sessionId) {
            sessionId = 'kantan_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            sessionStorage.setItem('kantan_session_id', sessionId);
        }
        return sessionId;
    }

    // Strip tracking parameters from URL
    function cleanUrl(url) {
        try {
            const urlObj = new URL(url);
            const paramsToRemove = [
                'fbclid', 'gclid', 'msclkid', 'mc_cid', 'mc_eid',
                '_ga', '_gl', '_ke', 'igshid',
                'utm_source', 'utm_medium', 'utm_campaign', 'utm_term', 'utm_content',
                'utm_id', 'utm_source_platform', 'utm_creative_format', 'utm_marketing_tactic'
            ];

            // Remove tracking parameters
            paramsToRemove.forEach(param => {
                urlObj.searchParams.delete(param);
            });

            // Return clean URL
            let cleanUrl = urlObj.origin + urlObj.pathname;

            // Add back search params if any remain
            const remainingParams = urlObj.searchParams.toString();
            if (remainingParams) {
                cleanUrl += '?' + remainingParams;
            }

            return cleanUrl;
        } catch (e) {
            // If URL parsing fails, return original
            return url;
        }
    }

    // Determine traffic source
    function getTrafficSource() {
        const referrer = document.referrer;
        if (!referrer) return 'direct';

        try {
            const referrerHost = new URL(referrer).hostname;
            const currentHost = window.location.hostname;

            // Same domain = direct/internal
            if (referrerHost === currentHost) return 'direct';

            // Search engines
            const searchEngines = ['google', 'bing', 'yahoo', 'duckduckgo', 'baidu'];
            for (let engine of searchEngines) {
                if (referrerHost.includes(engine)) return 'organic';
            }

            // Social media
            const socialMedia = ['facebook', 'twitter', 'instagram', 'linkedin', 'pinterest', 'tiktok'];
            for (let social of socialMedia) {
                if (referrerHost.includes(social)) return 'social';
            }
        } catch (e) {
            return 'referral';
        }

        // Everything else is referral
        return 'referral';
    }

    // Calculate 1-9 grid region from percentages
    function getClickRegion(xPct, yPct) {
        const col = xPct < 33 ? 0 : (xPct < 66 ? 1 : 2);
        const row = yPct < 33 ? 0 : (yPct < 66 ? 1 : 2);
        return row * 3 + col + 1; // Returns 1-9
    }

    // Throttle function for scroll tracking
    function throttle(func, limit) {
        let inThrottle;
        return function() {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }

    // Track page visit
    let startTime = Date.now();
    let pageViewCount = parseInt(sessionStorage.getItem('kantan_page_views') || '0');
    pageViewCount++;
    sessionStorage.setItem('kantan_page_views', pageViewCount.toString());

    // Current page sequence number (1-indexed)
    const sequenceNum = pageViewCount;

    // Determine if bounce (only one page viewed in session)
    const isBounce = pageViewCount === 1 ? 1 : 0;

    // Click tracking buffer
    let clickBuffer = [];

    // Scroll depth tracking (0-100%)
    let maxScrollDepth = 0;

    // Send tracking data (original page view tracking)
    function sendTrackingData() {
        const duration = Math.floor((Date.now() - startTime) / 1000); // seconds

        const data = new FormData();
        data.append('action', 'kantan_track_visit');
        data.append('session_id', getSessionId());
        data.append('page_url', cleanUrl(window.location.href));
        data.append('page_title', document.title);
        data.append('referrer', document.referrer);
        data.append('traffic_source', getTrafficSource());
        data.append('is_bounce', isBounce);
        data.append('visit_duration', duration);
        data.append('screen_width', window.screen.width);
        data.append('screen_height', window.screen.height);
        data.append('sequence_num', sequenceNum);
        data.append('scroll_depth', maxScrollDepth);

        // Use sendBeacon for reliability on page unload, fallback to fetch
        if (navigator.sendBeacon) {
            navigator.sendBeacon(kantanAnalytics.ajaxUrl, data);
        } else {
            fetch(kantanAnalytics.ajaxUrl, {
                method: 'POST',
                body: data,
                keepalive: true
            });
        }
    }

    // Send click tracking data
    function sendClickData() {
        if (clickBuffer.length === 0) return;

        const data = new FormData();
        data.append('action', 'kantan_track_clicks');
        data.append('session_id', getSessionId());
        data.append('page_url', cleanUrl(window.location.href));
        data.append('clicks', JSON.stringify(clickBuffer));

        // Clear buffer before sending
        const clicksToSend = clickBuffer;
        clickBuffer = [];

        if (navigator.sendBeacon) {
            navigator.sendBeacon(kantanAnalytics.ajaxUrl, data);
        } else {
            fetch(kantanAnalytics.ajaxUrl, {
                method: 'POST',
                body: data,
                keepalive: true
            });
        }
    }

    // Track clicks
    document.addEventListener('click', function(e) {
        const docHeight = Math.max(
            document.body.scrollHeight,
            document.documentElement.scrollHeight
        );
        const xPct = Math.round((e.clientX / window.innerWidth) * 100);
        const yPct = Math.round(((e.clientY + window.scrollY) / docHeight) * 100);

        // Get element text (truncated)
        let elementText = '';
        if (e.target.innerText) {
            elementText = e.target.innerText.substring(0, 100).trim();
        } else if (e.target.alt) {
            elementText = e.target.alt.substring(0, 100);
        } else if (e.target.title) {
            elementText = e.target.title.substring(0, 100);
        }

        clickBuffer.push({
            x: Math.min(Math.max(xPct, 0), 100),
            y: Math.min(Math.max(yPct, 0), 100),
            region: getClickRegion(xPct, yPct),
            tag: e.target.tagName.toLowerCase(),
            text: elementText
        });
    }, true); // Capture phase to catch all clicks

    // Track scroll depth
    window.addEventListener('scroll', throttle(function() {
        const scrollTop = window.scrollY || document.documentElement.scrollTop;
        const docHeight = Math.max(
            document.body.scrollHeight,
            document.documentElement.scrollHeight
        ) - window.innerHeight;

        if (docHeight > 0) {
            const depth = Math.round((scrollTop / docHeight) * 100);
            maxScrollDepth = Math.max(maxScrollDepth, Math.min(depth, 100));
        }
    }, 500));

    // Send initial tracking on page load
    setTimeout(sendTrackingData, 1000);

    // Update tracking before user leaves
    window.addEventListener('beforeunload', function() {
        sendTrackingData();
        sendClickData();
    });

    // Update page view data every 30 seconds
    setInterval(sendTrackingData, 30000);

    // Send click data every 5 seconds if buffer has data
    setInterval(sendClickData, 5000);

})();
