<?php
/**
 * Plugin Name: Fix ACF Sync
 * Plugin URI: https://plugins.joeljenkins.me
 * Description: Resolves ACF/ACFE sync conflicts after pulling a production database by clearing database-stored field groups, forcing ACF to load from your local PHP files instead.
 * Version: 1.1.0
 * Author: Kinda Useful Plugins
 * Author URI: https://plugins.joeljenkins.me
 * Requires at least: 5.0
 * Requires PHP: 7.0
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: fix-acf-sync
 */

if (!defined('ABSPATH')) {
	exit;
}

// Auto-updates from Kinda Useful Plugins
require_once plugin_dir_path(__FILE__) . 'includes/class-plugin-updater.php';
new Kinda_Useful_Plugin_Updater(
	'https://plugins.joeljenkins.me',
	__FILE__,
	'fix-acf-sync'
);

class Fix_ACF_Sync {

	private static $instance = null;

	public static function get_instance() {
		if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action('admin_menu', [$this, 'add_admin_menu']);
		add_action('admin_init', [$this, 'handle_clear_action']);
		add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);
	}

	public function add_admin_menu() {
		add_management_page(
			'Fix ACF Sync',
			'Fix ACF Sync',
			'manage_options',
			'fix-acf-sync',
			[$this, 'render_admin_page']
		);
	}

	public function enqueue_styles($hook) {
		if ($hook !== 'tools_page_fix-acf-sync') {
			return;
		}

		wp_add_inline_style('wp-admin', '
			.fix-acf-sync-wrap { max-width: 800px; }
			.fix-acf-sync-wrap h1 { margin-bottom: 20px; }
			.fix-acf-sync-box {
				background: #fff;
				padding: 20px 25px;
				border: 1px solid #ccd0d4;
				border-radius: 4px;
				margin-bottom: 20px;
			}
			.fix-acf-sync-box h2 {
				margin-top: 0;
				padding-bottom: 10px;
				border-bottom: 1px solid #eee;
			}
			.fix-acf-sync-warning {
				background: #fff8e5;
				border-color: #ffb900;
				border-left-width: 4px;
			}
			.fix-acf-sync-warning h2 { color: #826200; }
			.fix-acf-sync-info {
				background: #f0f6fc;
				border-color: #72aee6;
			}
			.fix-acf-sync-counts {
				display: flex;
				gap: 20px;
				margin: 15px 0;
			}
			.fix-acf-sync-count {
				background: #f0f0f1;
				padding: 15px 20px;
				border-radius: 4px;
				text-align: center;
				flex: 1;
			}
			.fix-acf-sync-count strong {
				display: block;
				font-size: 24px;
				color: #1d2327;
			}
			.fix-acf-sync-count span {
				color: #646970;
				font-size: 13px;
			}
			.fix-acf-sync-actions {
				margin-top: 20px;
				padding-top: 15px;
				border-top: 1px solid #eee;
			}
			.fix-acf-sync-actions .button-primary {
				background: #d63638;
				border-color: #d63638;
			}
			.fix-acf-sync-actions .button-primary:hover {
				background: #b32d2e;
				border-color: #b32d2e;
			}
			.fix-acf-sync-workflow {
				margin: 15px 0;
				padding-left: 20px;
			}
			.fix-acf-sync-workflow li {
				margin-bottom: 8px;
				color: #50575e;
			}
			.fix-acf-sync-workflow code {
				background: #f0f0f1;
				padding: 2px 6px;
				border-radius: 3px;
			}
		');
	}

	public function handle_clear_action() {
		if (!isset($_POST['fix_acf_sync_action']) || $_POST['fix_acf_sync_action'] !== 'clear') {
			return;
		}

		if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'fix_acf_sync_clear')) {
			wp_die('Security check failed.');
		}

		if (!current_user_can('manage_options')) {
			wp_die('You do not have permission to perform this action.');
		}

		global $wpdb;

		// Count before deletion
		$field_groups_count = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'acf-field-group'"
		);
		$fields_count = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'acf-field'"
		);

		// Delete ACF field groups
		$wpdb->delete($wpdb->posts, ['post_type' => 'acf-field-group']);

		// Delete ACF fields
		$wpdb->delete($wpdb->posts, ['post_type' => 'acf-field']);

		// Clear ACF caches
		wp_cache_flush();

		// Check if auto-sync was requested
		$auto_sync = isset($_POST['fix_acf_sync_auto_sync']) && $_POST['fix_acf_sync_auto_sync'] === '1';
		$synced_count = 0;

		if ($auto_sync) {
			$synced_count = $this->sync_local_json_to_database();
		}

		// Store results for display
		set_transient('fix_acf_sync_result', [
			'field_groups' => $field_groups_count,
			'fields' => $fields_count,
			'synced' => $synced_count,
			'auto_sync' => $auto_sync,
		], 60);

		// Redirect to prevent resubmission
		wp_redirect(add_query_arg([
			'page' => 'fix-acf-sync',
			'cleared' => '1',
		], admin_url('tools.php')));
		exit;
	}

	/**
	 * Sync all local field groups (JSON and PHP) to the database.
	 *
	 * @return int Number of field groups synced.
	 */
	private function sync_local_json_to_database() {
		// Check if ACF functions exist
		if (!function_exists('acf_import_field_group')) {
			return 0;
		}

		$synced = 0;
		$imported_keys = [];

		// 1. Import from ACF JSON files (acf-json/)
		if (function_exists('acf_get_local_json_files')) {
			$json_files = acf_get_local_json_files();

			foreach ($json_files as $key => $file) {
				$json = file_get_contents($file);
				if ($json === false) {
					continue;
				}

				$field_group = json_decode($json, true);
				if (!is_array($field_group) || empty($field_group['key'])) {
					continue;
				}

				// Import the field group
				acf_import_field_group($field_group);
				$imported_keys[] = $field_group['key'];
				$synced++;
			}
		}

		// 2. Import from ACFE PHP files (acfe-php/)
		// These are already loaded into ACF's local store, we just need to import them to DB
		if (function_exists('acfe_get_local_php_files')) {
			$php_files = acfe_get_local_php_files();

			foreach ($php_files as $key => $file) {
				// Skip if already imported from JSON
				if (in_array($key, $imported_keys, true)) {
					continue;
				}

				// Get the field group from ACF's local store
				if (function_exists('acf_get_local_field_group')) {
					$field_group = acf_get_local_field_group($key);
					if ($field_group) {
						// Get all fields for this group
						if (function_exists('acf_get_local_fields')) {
							$field_group['fields'] = acf_get_local_fields($key);
						}
						acf_import_field_group($field_group);
						$imported_keys[] = $key;
						$synced++;
					}
				}
			}
		}

		return $synced;
	}

	public function render_admin_page() {
		if (!current_user_can('manage_options')) {
			return;
		}

		global $wpdb;

		// Check for success message
		$cleared = isset($_GET['cleared']) && $_GET['cleared'] === '1';
		$result = get_transient('fix_acf_sync_result');
		if ($cleared && $result) {
			delete_transient('fix_acf_sync_result');
		}

		// Get current counts
		$field_groups_count = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'acf-field-group'"
		);
		$fields_count = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'acf-field'"
		);

		echo '<div class="wrap fix-acf-sync-wrap">';
		echo '<h1>Fix ACF Sync</h1>';

		// Success message
		if ($cleared && $result) {
			echo '<div class="notice notice-success is-dismissible">';
			echo '<p><strong>ACF database records cleared successfully!</strong></p>';
			echo '<p>Deleted ' . esc_html($result['field_groups']) . ' field group(s) and ' . esc_html($result['fields']) . ' field(s) from the database.</p>';
			if (!empty($result['auto_sync']) && $result['synced'] > 0) {
				echo '<p>Synced ' . esc_html($result['synced']) . ' field group(s) from local files to the database.</p>';
			} elseif (!empty($result['auto_sync']) && $result['synced'] === 0) {
				echo '<p>Auto-sync was enabled but no local files were found to import.</p>';
			} else {
				echo '<p>ACF will now load field definitions from your local sync files.</p>';
			}
			echo '</div>';
		}

		// Warning box
		echo '<div class="fix-acf-sync-box fix-acf-sync-warning">';
		echo '<h2>&#9888; When to Use This Tool</h2>';
		echo '<p>Use this tool <strong>immediately after pulling a production database</strong> to your local environment.</p>';
		echo '<p>When you pull a production database, ACF field groups stored in the database have different <code>modified</code> timestamps than your local sync files. ACF treats the database version as authoritative, which can cause:</p>';
		echo '<ul style="list-style: disc; margin-left: 20px;">';
		echo '<li>Your local sync files (<code>acf-json/</code> or <code>acfe-php/</code>) to be ignored</li>';
		echo '<li>The site to display old (production) field configurations</li>';
		echo '<li>Clicking "Save" on a field group to <strong>overwrite your local files</strong> with the old DB version</li>';
		echo '</ul>';
		echo '</div>';

		// Info box
		echo '<div class="fix-acf-sync-box fix-acf-sync-info">';
		echo '<h2>Recommended Workflow</h2>';
		echo '<ol class="fix-acf-sync-workflow">';
		echo '<li>Pull the production database</li>';
		echo '<li><strong>Immediately</strong> use this tool to clear ACF data</li>';
		echo '<li>Refresh your site</li>';
		echo '<li>Verify ACF is loading from your local sync files</li>';
		echo '<li>Continue development</li>';
		echo '</ol>';
		echo '</div>';

		// Current status box
		echo '<div class="fix-acf-sync-box">';
		echo '<h2>Current Database Status</h2>';
		echo '<div class="fix-acf-sync-counts">';
		echo '<div class="fix-acf-sync-count">';
		echo '<strong>' . esc_html($field_groups_count) . '</strong>';
		echo '<span>Field Groups</span>';
		echo '</div>';
		echo '<div class="fix-acf-sync-count">';
		echo '<strong>' . esc_html($fields_count) . '</strong>';
		echo '<span>Individual Fields</span>';
		echo '</div>';
		echo '</div>';

		if ($field_groups_count > 0 || $fields_count > 0) {
			echo '<div class="fix-acf-sync-actions">';
			echo '<form method="post">';
			echo '<input type="hidden" name="fix_acf_sync_action" value="clear">';
			wp_nonce_field('fix_acf_sync_clear');
			echo '<p><strong>This action will delete all ACF field groups and fields from the database.</strong></p>';
			echo '<p>After clearing, ACF will reload field definitions from your local JSON files on the next page load.</p>';
			echo '<p style="margin: 15px 0;">';
			echo '<label><input type="checkbox" name="fix_acf_sync_auto_sync" value="1" checked> ';
			echo '<strong>Auto-sync from local files</strong> &mdash; Immediately import all field groups from <code>acf-json/</code> and <code>acfe-php/</code> into the database after clearing</label>';
			echo '</p>';
			echo '<p><button type="submit" class="button button-primary" onclick="return confirm(\'Are you sure you want to delete all ACF field groups and fields from the database? This cannot be undone.\');">Clear ACF Database Records</button></p>';
			echo '</form>';
			echo '</div>';
		} else {
			echo '<p style="color: #00a32a; margin-top: 15px;"><strong>&#10003; No ACF records in the database.</strong> ACF is loading from local sync files.</p>';
		}

		echo '</div>';
		echo '</div>';
	}
}

// Initialize the plugin
Fix_ACF_Sync::get_instance();
