<?php
/**
 * Plugin Name: ACF Reports
 * Plugin URI: https://plugins.joeljenkins.me
 * Description: Generates detailed reports on ACF Flexible Content block usage across your WordPress site. Quickly see which blocks are used, how many times, and on which pages - with CSV export for deeper analysis.
 * Version: 1.0.1
 * Author: Kinda Useful Plugins
 * Author URI: https://plugins.joeljenkins.me
 * Requires at least: 5.0
 * Requires PHP: 7.0
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: acf-reports
 */

if (!defined('ABSPATH')) {
  exit;
}

// Auto-updates from Kinda Useful Plugins
require_once plugin_dir_path(__FILE__) . 'includes/class-plugin-updater.php';
new Kinda_Useful_Plugin_Updater(
    'https://plugins.joeljenkins.me',
    __FILE__,
    'acf-reports'
);

class ACF_Reports {

  private static $instance = null;

  public static function get_instance() {
    if (null === self::$instance) {
      self::$instance = new self();
    }
    return self::$instance;
  }

  private function __construct() {
    add_action('admin_menu', [$this, 'add_admin_menu']);
    add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);
    add_action('admin_init', [$this, 'handle_csv_export']);
  }

  public function add_admin_menu() {
    add_menu_page(
      'ACF Reports',
      'ACF Reports',
      'manage_options',
      'acf-reports',
      [$this, 'render_admin_page'],
      'dashicons-chart-bar',
      80
    );
  }

  public function handle_csv_export() {
    if (!isset($_GET['page']) || $_GET['page'] !== 'acf-reports') {
      return;
    }
    if (!isset($_GET['export']) || $_GET['export'] !== 'csv') {
      return;
    }
    if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'acf_reports_export')) {
      return;
    }
    if (!current_user_can('manage_options')) {
      return;
    }
    if (!isset($_GET['field_group']) || strpos($_GET['field_group'], '|') === false) {
      return;
    }

    list($group_key, $field_name) = explode('|', $_GET['field_group']);

    // Get layouts for labels
    $field_groups = acf_get_field_groups();
    $available_blocks = [];

    foreach ($field_groups as $group) {
      if ($group['key'] !== $group_key) {
        continue;
      }
      $fields = acf_get_fields($group['key']);
      if ($fields) {
        foreach ($fields as $field) {
          if ($field['type'] === 'flexible_content' && $field['name'] === $field_name) {
            foreach ($field['layouts'] as $layout) {
              $available_blocks[$layout['name']] = $layout['label'];
            }
            break 2;
          }
        }
      }
    }

    // Get block usage
    $block_usage = $this->get_block_usage($field_name);

    // Sort by total usage
    uasort($block_usage, function($a, $b) {
      return $b['total'] - $a['total'];
    });

    // Generate CSV
    $filename = 'acf-block-usage-' . date('Y-m-d') . '.csv';

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');

    $output = fopen('php://output', 'w');

    // Header row
    fputcsv($output, ['Block Name', 'Block Label', 'Total Uses', 'Page Title', 'Page URL', 'Edit URL', 'Instances on Page']);

    foreach ($block_usage as $block_name => $data) {
      $label = isset($available_blocks[$block_name]) ? $available_blocks[$block_name] : $block_name;

      foreach ($data['pages'] as $page) {
        fputcsv($output, [
          $block_name,
          $label,
          $data['total'],
          $page['title'],
          get_permalink($page['id']),
          get_edit_post_link($page['id'], 'raw'),
          $page['count']
        ]);
      }
    }

    fclose($output);
    exit;
  }

  public function enqueue_styles($hook) {
    if ($hook !== 'toplevel_page_acf-reports') {
      return;
    }

    wp_add_inline_style('wp-admin', '
      .acf-reports-wrap { max-width: 1200px; }
      .acf-reports-wrap h1 { margin-bottom: 20px; }
      .acf-reports-summary {
        background: #fff;
        padding: 20px;
        border: 1px solid #ccd0d4;
        margin-bottom: 20px;
        border-radius: 4px;
      }
      .acf-reports-summary h2 { margin-top: 0; }
      .acf-reports-table {
        background: #fff;
        border-collapse: collapse;
        width: 100%;
        border: 1px solid #ccd0d4;
      }
      .acf-reports-table th {
        background: #f1f1f1;
        text-align: left;
        padding: 12px 15px;
        border-bottom: 1px solid #ccd0d4;
        font-weight: 600;
      }
      .acf-reports-table td {
        padding: 12px 15px;
        border-bottom: 1px solid #e5e5e5;
        vertical-align: top;
      }
      .acf-reports-table tr:hover td { background: #f9f9f9; }
      .acf-reports-table .block-name { font-weight: 600; font-family: monospace; }
      .acf-reports-table .count-cell { text-align: center; font-weight: 600; }
      .acf-reports-table .pages-list { margin: 0; padding: 0; list-style: none; }
      .acf-reports-table .pages-list li {
        padding: 4px 0;
        border-bottom: 1px dashed #e5e5e5;
      }
      .acf-reports-table .pages-list li:last-child { border-bottom: none; }
      .acf-reports-table .page-count {
        background: #2271b1;
        color: #fff;
        padding: 2px 8px;
        border-radius: 10px;
        font-size: 11px;
        margin-left: 8px;
      }
      .acf-reports-unused { margin-top: 30px; }
      .acf-reports-unused h3 { margin-bottom: 15px; }
      .unused-blocks-list {
        background: #fff;
        padding: 15px 20px;
        border: 1px solid #ccd0d4;
        border-radius: 4px;
      }
      .unused-blocks-list code {
        background: #f0f0f1;
        padding: 3px 6px;
        margin: 3px;
        display: inline-block;
        border-radius: 3px;
      }
      .pages-accordion {
        cursor: pointer;
        user-select: none;
      }
      .pages-accordion::before {
        content: "▶";
        display: inline-block;
        margin-right: 8px;
        font-size: 10px;
        transition: transform 0.2s;
      }
      .pages-accordion.open::before {
        transform: rotate(90deg);
      }
      .pages-list.collapsed {
        display: none;
      }
      .field-group-selector {
        background: #fff;
        padding: 20px;
        border: 1px solid #ccd0d4;
        margin-bottom: 20px;
        border-radius: 4px;
      }
      .field-group-selector select {
        min-width: 300px;
        padding: 8px;
      }
      .field-group-selector .button {
        margin-left: 10px;
      }
      .export-button {
        margin-left: 15px;
      }
    ');
  }

  public function render_admin_page() {
    if (!current_user_can('manage_options')) {
      return;
    }

    // Check if ACF is active
    if (!function_exists('acf_get_field_groups')) {
      echo '<div class="wrap acf-reports-wrap">';
      echo '<h1>ACF Reports</h1>';
      echo '<div class="notice notice-error"><p>Advanced Custom Fields Pro is required for this plugin.</p></div>';
      echo '</div>';
      return;
    }

    // Get all field groups
    $field_groups = acf_get_field_groups();
    $flexible_groups = [];

    foreach ($field_groups as $group) {
      $fields = acf_get_fields($group['key']);
      if ($fields) {
        foreach ($fields as $field) {
          if ($field['type'] === 'flexible_content') {
            $flexible_groups[] = [
              'group_key' => $group['key'],
              'group_title' => $group['title'],
              'field_name' => $field['name'],
              'field_key' => $field['key'],
              'layouts' => $field['layouts'] ?? []
            ];
          }
        }
      }
    }

    // Get selected field group
    $selected_group = isset($_GET['field_group']) ? sanitize_text_field($_GET['field_group']) : '';
    $selected_field = isset($_GET['field_name']) ? sanitize_text_field($_GET['field_name']) : '';

    echo '<div class="wrap acf-reports-wrap">';
    echo '<h1>ACF Flexible Content Block Usage Report</h1>';

    // Field group selector
    echo '<div class="field-group-selector">';
    echo '<form method="get">';
    echo '<input type="hidden" name="page" value="acf-reports">';
    echo '<label><strong>Select Field Group:</strong> </label>';
    echo '<select name="field_group" onchange="this.form.submit()">';
    echo '<option value="">-- Select a Flexible Content Field --</option>';

    foreach ($flexible_groups as $fg) {
      $value = $fg['group_key'] . '|' . $fg['field_name'];
      $selected = ($selected_group === $fg['group_key'] && $selected_field === $fg['field_name']) ? 'selected' : '';
      echo '<option value="' . esc_attr($value) . '" ' . $selected . '>';
      echo esc_html($fg['group_title'] . ' - ' . $fg['field_name']);
      echo '</option>';
    }

    echo '</select>';
    echo '</form>';
    echo '</div>';

    if (isset($_GET['field_group']) && strpos($_GET['field_group'], '|') !== false) {
      list($selected_group, $selected_field) = explode('|', $_GET['field_group']);
      $this->render_report($selected_group, $selected_field, $flexible_groups);
    } elseif (!empty($flexible_groups)) {
      echo '<p>Please select a flexible content field group above to view the usage report.</p>';
    } else {
      echo '<div class="notice notice-warning"><p>No flexible content fields found.</p></div>';
    }

    echo '</div>';
  }

  private function render_report($group_key, $field_name, $flexible_groups) {
    // Find the selected group's layouts
    $layouts = [];
    foreach ($flexible_groups as $fg) {
      if ($fg['group_key'] === $group_key && $fg['field_name'] === $field_name) {
        $layouts = $fg['layouts'];
        break;
      }
    }

    // Get all available layout names
    $available_blocks = [];
    foreach ($layouts as $layout) {
      $available_blocks[$layout['name']] = $layout['label'];
    }

    // Query the database for block usage
    $block_usage = $this->get_block_usage($field_name);

    // Sort by total usage (most used first)
    uasort($block_usage, function($a, $b) {
      return $b['total'] - $a['total'];
    });

    // Calculate totals
    $total_blocks_used = count($block_usage);
    $total_instances = array_sum(array_column($block_usage, 'total'));
    $total_available = count($available_blocks);
    $unused_blocks = array_diff_key($available_blocks, $block_usage);

    // Summary
    echo '<div class="acf-reports-summary">';
    echo '<h2>Summary</h2>';
    echo '<p><strong>Total Available Blocks:</strong> ' . $total_available . '</p>';
    echo '<p><strong>Blocks In Use:</strong> ' . $total_blocks_used . '</p>';
    echo '<p><strong>Total Block Instances:</strong> ' . $total_instances . '</p>';
    echo '<p><strong>Unused Blocks:</strong> ' . count($unused_blocks) . '</p>';

    // Export button
    $export_url = add_query_arg([
      'page' => 'acf-reports',
      'field_group' => $_GET['field_group'],
      'export' => 'csv',
      '_wpnonce' => wp_create_nonce('acf_reports_export')
    ], admin_url('admin.php'));
    echo '<p style="margin-top:15px;"><a href="' . esc_url($export_url) . '" class="button button-primary">Export CSV</a></p>';

    echo '</div>';

    // Usage table
    if (!empty($block_usage)) {
      echo '<table class="acf-reports-table">';
      echo '<thead>';
      echo '<tr>';
      echo '<th style="width:28%;">Block Name</th>';
      echo '<th style="width:28%;">Label</th>';
      echo '<th style="width:10%;" class="count-cell">Total Uses</th>';
      echo '<th style="width:34%;">Pages Using This Block</th>';
      echo '</tr>';
      echo '</thead>';
      echo '<tbody>';

      foreach ($block_usage as $block_name => $data) {
        $label = isset($available_blocks[$block_name]) ? $available_blocks[$block_name] : $block_name;

        echo '<tr>';
        echo '<td class="block-name">' . esc_html($block_name) . '</td>';
        echo '<td>' . esc_html($label) . '</td>';
        echo '<td class="count-cell">' . $data['total'] . '</td>';
        echo '<td>';

        $page_count = count($data['pages']);
        $page_label = $page_count === 1 ? '1 page' : $page_count . ' pages';
        echo '<div class="pages-accordion" onclick="this.classList.toggle(\'open\'); this.nextElementSibling.classList.toggle(\'collapsed\');">';
        echo $page_label;
        echo '</div>';
        echo '<ul class="pages-list collapsed">';

        foreach ($data['pages'] as $page) {
          $edit_link = get_edit_post_link($page['id']);
          $view_link = get_permalink($page['id']);
          echo '<li>';
          echo '<a href="' . esc_url($edit_link) . '">' . esc_html($page['title']) . '</a>';
          echo ' <a href="' . esc_url($view_link) . '" target="_blank" title="View page">(view)</a>';
          echo '<span class="page-count">' . $page['count'] . 'x</span>';
          echo '</li>';
        }

        echo '</ul>';
        echo '</td>';
        echo '</tr>';
      }

      echo '</tbody>';
      echo '</table>';
    } else {
      echo '<div class="notice notice-warning"><p>No block usage found for this field.</p></div>';
    }

    // Unused blocks
    if (!empty($unused_blocks)) {
      echo '<div class="acf-reports-unused">';
      echo '<h3>Unused Blocks (' . count($unused_blocks) . ')</h3>';
      echo '<div class="unused-blocks-list">';

      foreach ($unused_blocks as $name => $label) {
        echo '<code title="' . esc_attr($label) . '">' . esc_html($name) . '</code> ';
      }

      echo '</div>';
      echo '</div>';
    }
  }

  private function get_block_usage($field_name) {
    global $wpdb;

    $block_usage = [];

    // Debug: Show what we're looking for
    $debug = isset($_GET['debug']) && $_GET['debug'] === '1';

    if ($debug) {
      echo '<div style="background:#fff;padding:20px;margin:20px 0;border:2px solid red;">';
      echo '<h3>Debug Info</h3>';
      echo '<p><strong>Field name:</strong> ' . esc_html($field_name) . '</p>';

      // Check the raw meta value for the blocks field
      $sample_value = $wpdb->get_row($wpdb->prepare("
        SELECT post_id, meta_value
        FROM {$wpdb->postmeta}
        WHERE meta_key = %s
        AND meta_value != ''
        LIMIT 1
      ", $field_name));

      if ($sample_value) {
        echo '<p><strong>Sample raw meta_value for "' . esc_html($field_name) . '" (post_id: ' . $sample_value->post_id . '):</strong></p>';
        echo '<pre style="max-height:300px;overflow:auto;">';
        $unserialized = maybe_unserialize($sample_value->meta_value);
        print_r($unserialized);
        echo '</pre>';

        // Also try get_field to see what ACF returns
        echo '<p><strong>get_field() result for post ' . $sample_value->post_id . ':</strong></p>';
        echo '<pre style="max-height:300px;overflow:auto;">';
        $acf_value = get_field($field_name, $sample_value->post_id);
        if ($acf_value) {
          echo 'Type: ' . gettype($acf_value) . "\n";
          echo 'Count: ' . (is_array($acf_value) ? count($acf_value) : 'N/A') . "\n";
          if (is_array($acf_value) && !empty($acf_value[0])) {
            echo 'First item keys: ' . implode(', ', array_keys($acf_value[0])) . "\n";
          }
        } else {
          echo 'NULL or empty';
        }
        echo '</pre>';
      }
      echo '</div>';
    }

    // ACF stores the layout names as a serialized array in the main field meta
    // Query for all posts with this field
    $results = $wpdb->get_results($wpdb->prepare("
      SELECT pm.post_id, pm.meta_value
      FROM {$wpdb->postmeta} pm
      INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
      WHERE pm.meta_key = %s
      AND pm.meta_value != ''
      AND p.post_status = 'publish'
      AND p.post_type NOT IN ('revision', 'acf-field', 'acf-field-group')
    ", $field_name));

    // Process each post's blocks
    foreach ($results as $row) {
      $post_id = $row->post_id;
      $layouts = maybe_unserialize($row->meta_value);

      if (!is_array($layouts) || empty($layouts)) {
        continue;
      }

      $post = get_post($post_id);
      if (!$post) {
        continue;
      }

      // Count layouts for this post
      $layout_counts = array_count_values($layouts);

      foreach ($layout_counts as $layout_name => $count) {
        if (!is_string($layout_name) || empty($layout_name)) {
          continue;
        }

        if (!isset($block_usage[$layout_name])) {
          $block_usage[$layout_name] = [
            'total' => 0,
            'pages' => []
          ];
        }

        $block_usage[$layout_name]['total'] += $count;
        $block_usage[$layout_name]['pages'][] = [
          'id' => $post_id,
          'title' => $post->post_title,
          'count' => $count
        ];
      }
    }

    // Sort pages within each block by count (most instances first)
    foreach ($block_usage as &$data) {
      usort($data['pages'], function($a, $b) {
        return $b['count'] - $a['count'];
      });
    }

    return $block_usage;
  }
}

// Initialize the plugin
ACF_Reports::get_instance();
